<?php
// Include the FPDF library
require('../fpdf186/fpdf.php');
require_once '../vendor/phpqrcode/qrlib.php';
require_once '../config/app.php';
session_start();
require_once '../models/db_connection.php';
require_once '../models/student_model.php';
require_once '../models/exam_model.php';
require_once '../models/course_model.php';

$course =  new Course($conn);
$coursesData =  $course->getAllCourses();
$student =  new Student($conn);
$studentsData = $student->getAllStudents();
$exam =  new Exam($conn);

function generateRegistrationNumber(array $studentValue): string {
    $existing = trim((string)($studentValue['en_id'] ?? ''));
    if ($existing !== '') {
        return $existing;
    }

    $defaultRegionCode = '025';
    $regionDigits = preg_replace('/\D/', '', (string)($studentValue['region_code'] ?? $defaultRegionCode));
    $regionCode = str_pad(substr($regionDigits, 0, 3), 3, '0', STR_PAD_LEFT);

    $courseInitial = (string)($studentValue['course_initial'] ?? '');
    if ($courseInitial === '' && !empty($studentValue['course_name'])) {
        $parts = preg_split('/\s+/', strtoupper($studentValue['course_name']));
        $courseInitial = implode('', array_map(function ($word) {
            return $word !== '' ? $word[0] : '';
        }, $parts));
    }
    $courseInitial = strtoupper(preg_replace('/[^A-Z]/', '', $courseInitial));
    if ($courseInitial === '') {
        $courseInitial = 'HGV';
    }

    // Prefer intake from student_course table
    $intakeRaw = $studentValue['intake'] ?? ($studentValue['intake_number'] ?? null);
    $intakeNumber = 1;
    if ($intakeRaw !== null) {
        $intakeDigits = preg_replace('/\D+/', '', (string)$intakeRaw);
        if ($intakeDigits !== '') {
            $intakeNumber = (int)$intakeDigits;
        } elseif (is_numeric($intakeRaw)) {
            $intakeNumber = (int)$intakeRaw;
        }
    }
    $intakeNumber = max(1, $intakeNumber);

    $studentNumberPart = str_pad((string)($studentValue['student_id'] ?? 0), 4, '0', STR_PAD_LEFT);

    $yearPart = '';
    if (!empty($studentValue['en_date'])) {
        try {
            $yearPart = (new DateTime($studentValue['en_date']))->format('y');
        } catch (Exception $e) {
            $yearPart = '';
        }
    }
    if ($yearPart === '' && !empty($studentValue['year'])) {
        $yearPart = substr((string)$studentValue['year'], -2);
    }
    if ($yearPart === '') {
        $yearPart = date('y');
    }

    return sprintf('M%s/%s/%d%s/%s', $regionCode, $courseInitial, $intakeNumber, $studentNumberPart, $yearPart);
}

function generateCertificateNumber(array $studentValue): string {
    $examPart = str_pad((string)($studentValue['exam_id'] ?? 0), 4, '0', STR_PAD_LEFT);
    $studentPart = str_pad((string)($studentValue['student_id'] ?? 0), 4, '0', STR_PAD_LEFT);
    $yearPart = date('y');
    return $yearPart . $examPart . $studentPart;
}



// Create a new PDF instance
$pdf = new FPDF();
// Register custom Lobster font (added by user)
$pdf->AddFont('Lobster-Regular','', 'Lobster-Regular.php');

// Base URL for public result view
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$publicResultUrlBase = $scheme . '://' . $appHost . '/public_result.php';

function addCertificateBarcode(FPDF $pdf, $data) {
    $tmpBase = tempnam(sys_get_temp_dir(), 'qr_');
    $tmpFile = $tmpBase . '.png';
    if (file_exists($tmpBase)) {
        unlink($tmpBase);
    }

    // Generate a small QR barcode using the bundled phpqrcode library
    QRcode::png($data, $tmpFile, QR_ECLEVEL_L, 4, 2);

    $qrSize = 30;
    $barcodeX = ($pdf->GetPageWidth() - $qrSize) / 2; // center horizontally
    $barcodeY = $pdf->GetPageHeight() - 58; // near bottom
    $pdf->Image($tmpFile, $barcodeX, $barcodeY, $qrSize, $qrSize, 'PNG');

    if (file_exists($tmpFile)) {
        unlink($tmpFile);
    }
}

function renderCertificate($pdf, $studentValue, $publicResultUrlBase) {
    $pdf->AddPage();

    // Set font for the text
    $pdf->SetFont('Arial', '', 16);

    // Add certificate image as background
    $pdf->Image('certificate_template.jpg', 0, 0, $pdf->GetPageWidth(), $pdf->GetPageHeight());

    // Student registration number (top area)
    $pdf->SetFont('Arial', 'B', 14);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->SetXY(10, 198); // Adjust X and Y coordinates to position the text
    $registrationNumber = generateRegistrationNumber($studentValue);
    $pdf->Cell(0, 10, 'REG NO ' . $registrationNumber, 0, 1, 'C');

    // Certificate number (relocated near the bottom)
    $pdf->SetFont('Arial', '', 10);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->SetXY(10, 266);
    $rawCertNumber = generateCertificateNumber($studentValue);
    $spacedCertNumber = trim(chunk_split($rawCertNumber, 4, ' ')); // Add generous spacing for readability
    $certNumber = 'CERT NO ' . $spacedCertNumber;
    $pdf->Cell(0, 10, $certNumber, 0, 1, 'C');

    // // Add Registration number
    // $pdf->SetTextColor(0, 0, 255);
    // $pdf->SetFont('Arial', '', 9);
    // $pdf->SetXY(0, 20); // Adjust X and Y coordinates to position the text
    // $pdf->Cell(0, 10, "REG-TRD/A.21/MWZ/2010/54" , 0, 1, 'C');

    // Add student image
    if (!empty($studentValue['filename']) && file_exists("../uploads/".$studentValue['filename'])) {
        $pdf->Image("../uploads/".$studentValue['filename'], 150, 26, 28, 35);
    }

    // Add student Name (use Lobster font for emphasis)
    $pdf->SetFont('Times', 'B', 42);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->SetXY(10, 132); // Adjust X and Y coordinates to position the text
    $pdf->Cell(0, 10,$studentValue['fullname'], 0, 1, 'C');

    // Add course name (centered beneath student name)
    if (!empty($studentValue['course_name'])) {
        $pdf->SetFont('Times', 'B', 26);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->SetXY(10, 178);
        $pdf->Cell(0, 10, strtoupper($studentValue['course_name']), 0, 1, 'C');


         //date of completion
    $pdf->SetFont('Times', '', 20);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->SetXY(10, 188); // Adjust X and Y coordinates to position the text
    $fromDate = !empty($studentValue['en_date']) ? (new DateTime($studentValue['en_date']))->format("d M, Y") : '';
    $toDate = date("d M, Y");
    $pdf->Cell(0, 10, 'From ' . $fromDate . ' To ' . $toDate, 0, 1, 'C');
    }

    // Signatures (images stored alongside this controller)
    $principalSignaturePath = __DIR__ . '/principal.png';
    if (file_exists($principalSignaturePath)) {
        $pdf->Image($principalSignaturePath, 24, 235, 25);
    }

    $directorSignaturePath = __DIR__ . '/director.png';
    if (file_exists($directorSignaturePath)) {
        $pdf->Image($directorSignaturePath, 146, 239, 40);
    }

    // // Coordinator
    // $pdf->SetFont('Times', 'B', 14);
    // $pdf->SetTextColor(0, 0, 0);
    // $pdf->SetXY(157, 247);
    // $pdf->Cell(0, 10, "Enock Msabila", 0, 1, '');

    // // Principal
    // if (!empty($studentValue['course_name'])) {
    //     $pdf->SetFont('Times', 'B', 14);
    //     $pdf->SetTextColor(0, 0, 0);
    //     $pdf->SetXY(26,245);
    //     $pdf->Cell(0, 10, "Jackson M Lupeja " , 0, 1, '');
    // }

   

    // Add QR barcode that links to the public results page
    $studentId = $studentValue['student_id'] ?? null;
    $examId = $studentValue['exam_id'] ?? null;
    if ($studentId) {
        $queryParams = ['student_id' => $studentId];
        if ($examId) {
            $queryParams['exam_id'] = $examId;
        }
        $publicUrl = $publicResultUrlBase . '?' . http_build_query($queryParams);
        addCertificateBarcode($pdf, $publicUrl);
    }
}

// If a single exam_id is provided, generate a certificate for that student only
if (isset($_GET['exam_id'])) {
    $examId = (int)$_GET['exam_id'];
    $singleQuery = mysqli_query($conn, "
        SELECT se.id AS exam_id, st.*, sc.en_date, sc.intake, c.name AS course_name
        FROM student_exams se
        JOIN students st ON st.student_id = se.student
        LEFT JOIN student_course sc ON sc.s_id = st.student_id
        LEFT JOIN courses c ON c.course_id = se.course
        WHERE se.id = '$examId' AND se.remark = 'passed'
        LIMIT 1
    ");
    $singleStudent = $singleQuery ? mysqli_fetch_assoc($singleQuery) : null;
    if ($singleStudent) {
        renderCertificate($pdf, $singleStudent, $publicResultUrlBase);
        $pdf->Output();
        exit;
    }
}

// Otherwise, fall back to session filters if set
$year = isset($_SESSION['year']) ? $_SESSION['year'] : null;
$courseID = isset($_SESSION['courseID']) ? $_SESSION['courseID'] : null;
$studentsData = [];
if ($year && $courseID) {
    $studentsData =  $course->getAllStudentsByCourseAndYear($courseID, $year);
}

// Generate certificates for filtered students (passed students only)
foreach ($studentsData as $studentValue) {
    $isEnrolled = $student->isStudentEnrolled($studentValue['student_id']);
    $studentRes = $exam->generateResults($studentValue['student_id']);
    if (!isset($studentRes['average']) || $studentRes['average'] < 70) {
      continue;
    }
    renderCertificate($pdf, $studentValue, $publicResultUrlBase);
}

// Output the PDF to the browser
$pdf->Output();


?>
