<?php
require_once __DIR__ . '/config/app.php';
require_once __DIR__ . '/models/db_connection.php';

$studentId = isset($_GET['student_id']) ? (int)$_GET['student_id'] : 0;
$examId = isset($_GET['exam_id']) ? (int)$_GET['exam_id'] : 0;

if ($studentId <= 0) {
    http_response_code(400);
    echo 'Student not specified.';
    exit;
}

// Fetch student details with course info
$studentSql = "
    SELECT st.*, c.name AS course_name, sc.en_date
    FROM students st
    LEFT JOIN student_course sc ON sc.s_id = st.student_id
    LEFT JOIN courses c ON c.course_id = sc.course_id
    WHERE st.student_id = ?
    LIMIT 1
";
$studentStmt = $conn->prepare($studentSql);
$studentStmt->bind_param("i", $studentId);
$studentStmt->execute();
$studentRes = $studentStmt->get_result();
$student = $studentRes->fetch_assoc();

// Fetch exam result (use specific exam if provided, otherwise latest passed)
$examSql = "
    SELECT se.*, c.name AS course_name
    FROM student_exams se
    LEFT JOIN courses c ON c.course_id = se.course
    WHERE se.student = ?
";
$params = [$studentId];
$types = "i";
if ($examId > 0) {
    $examSql .= " AND se.id = ? ";
    $params[] = $examId;
    $types .= "i";
} else {
    $examSql .= " AND se.remark = 'passed' ";
}
$examSql .= " ORDER BY se.id DESC LIMIT 1";

$examStmt = $conn->prepare($examSql);
$examStmt->bind_param($types, ...$params);
$examStmt->execute();
$examRes = $examStmt->get_result();
$exam = $examRes->fetch_assoc();

function safeVal($value) {
    return htmlspecialchars((string)$value, ENT_QUOTES, 'UTF-8');
}

// Compute average and breakdown if available
$average = null;
$marksBreakdown = [];
if ($exam) {
    if (isset($exam['average']) && is_numeric($exam['average'])) {
        $average = (float)$exam['average'];
    }
    if (!empty($exam['subject_marks'])) {
        $decoded = json_decode($exam['subject_marks'], true);
        if (is_array($decoded)) {
            $marksBreakdown = $decoded;
            if ($average === null && count($decoded) > 0) {
                $average = array_sum($decoded) / max(count($decoded), 1);
            }
        }
    }
    if ($average === null && isset($exam['practical']) && isset($exam['theory'])) {
        $avgParts = [];
        if (is_numeric($exam['practical'])) {
            $avgParts[] = (float)$exam['practical'];
        }
        if (is_numeric($exam['theory'])) {
            $avgParts[] = (float)$exam['theory'];
        }
        if (!empty($avgParts)) {
            $average = array_sum($avgParts) / count($avgParts);
            $marksBreakdown = [
                'Practical' => $avgParts[0] ?? null,
                'Theory' => $avgParts[1] ?? null,
            ];
        }
    }
}

$grade = '';
if ($average !== null) {
    if ($average >= 70) {
        $grade = 'Passed';
    } elseif ($average >= 50) {
        $grade = 'Retake';
    } else {
        $grade = 'Failed';
    }
}

$photoPath = (!empty($student['filename']) && file_exists(__DIR__ . '/uploads/' . $student['filename']))
    ? 'uploads/' . safeVal($student['filename'])
    : null;

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Result</title>
    <style>
        :root {
            --bg: #0f172a;
            --panel: rgba(255, 255, 255, 0.05);
            --card: rgba(255, 255, 255, 0.08);
            --text: #e2e8f0;
            --accent: #22d3ee;
            --muted: #94a3b8;
            --border: rgba(255, 255, 255, 0.12);
            --success: #34d399;
            --warn: #fbbf24;
            --fail: #f87171;
        }
        * { box-sizing: border-box; }
        body {
            margin: 0;
            font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
            color: var(--text);
            background: radial-gradient(circle at 20% 20%, rgba(34,211,238,0.15), transparent 25%),
                        radial-gradient(circle at 80% 0%, rgba(52,211,153,0.15), transparent 30%),
                        var(--bg);
            min-height: 100vh;
            padding: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .shell {
            width: min(1000px, 100%);
            background: var(--panel);
            border: 1px solid var(--border);
            border-radius: 18px;
            padding: 28px;
            backdrop-filter: blur(12px);
            box-shadow: 0 20px 60px rgba(0,0,0,0.35);
        }
        .header {
            display: flex;
            align-items: center;
            gap: 18px;
            margin-bottom: 20px;
        }
        .avatar {
            width: 88px;
            height: 88px;
            border-radius: 16px;
            border: 2px solid var(--border);
            object-fit: cover;
            background: var(--card);
        }
        .title-block h1 {
            margin: 0;
            font-size: 26px;
            letter-spacing: 0.5px;
        }
        .title-block p {
            margin: 2px 0 0;
            color: var(--muted);
        }
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 14px;
            margin-bottom: 16px;
        }
        .card {
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 14px;
            padding: 14px 16px;
        }
        .label { color: var(--muted); font-size: 12px; letter-spacing: 0.4px; }
        .value { font-size: 16px; margin-top: 4px; }
        .tag {
            display: inline-flex;
            align-items: center;
            padding: 6px 12px;
            border-radius: 999px;
            font-weight: 600;
            border: 1px solid var(--border);
            background: rgba(255,255,255,0.04);
        }
        .tag.success { color: var(--success); border-color: rgba(52,211,153,0.35); }
        .tag.warn { color: var(--warn); border-color: rgba(251,191,36,0.35); }
        .tag.fail { color: var(--fail); border-color: rgba(248,113,113,0.35); }
        .marks {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
            gap: 10px;
        }
        .marks .card { padding: 12px; }
        .empty {
            text-align: center;
            padding: 18px;
            color: var(--muted);
        }
        @media (max-width: 640px) {
            body { padding: 16px; }
            .shell { padding: 20px; }
            .header { flex-direction: column; align-items: flex-start; }
        }
    </style>
</head>
<body>
    <div class="shell">
        <div class="header">
            <?php if ($photoPath): ?>
                <img class="avatar" src="<?php echo $photoPath; ?>" alt="Student photo">
            <?php else: ?>
                <div class="avatar"></div>
            <?php endif; ?>
            <div class="title-block">
                <p class="label">Student</p>
                <h1><?php echo safeVal($student['fullname'] ?? 'Unknown Student'); ?></h1>
                <p><?php echo safeVal($student['en_id'] ?? ''); ?></p>
            </div>
        </div>

        <div class="grid">
            <div class="card">
                <div class="label">Course</div>
                <div class="value"><?php echo safeVal($student['course_name'] ?? ($exam['course_name'] ?? 'N/A')); ?></div>
            </div>
            <div class="card">
                <div class="label">Enrollment Date</div>
                <div class="value"><?php echo safeVal($student['en_date'] ?? ''); ?></div>
            </div>
            <div class="card">
                <div class="label">Identification</div>
                <div class="value"><?php echo safeVal($student['ident_number'] ?? ''); ?></div>
            </div>
            <div class="card">
                <div class="label">Average Score</div>
                <div class="value">
                    <?php echo $average !== null ? number_format($average, 2) : 'N/A'; ?>
                    <?php if ($grade): ?>
                        <span class="tag <?php echo strtolower($grade) === 'passed' ? 'success' : (strtolower($grade) === 'failed' ? 'fail' : 'warn'); ?>" style="margin-left:8px;"><?php echo safeVal($grade); ?></span>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <div class="card" style="margin-bottom: 12px;">
            <div class="label">Exam Summary</div>
            <div class="value" style="margin-top:6px;">
                <?php if ($exam): ?>
                    <?php
                        $dateVal = '';
                        if (!empty($exam['created_date'])) {
                            $dateVal = date('M j, Y', strtotime($exam['created_date']));
                        }
                    ?>
                    <div>Exam ID: <?php echo safeVal($exam['id'] ?? '-'); ?></div>
                    <?php if ($dateVal): ?><div>Date: <?php echo safeVal($dateVal); ?></div><?php endif; ?>
                    <?php if (!empty($exam['remark'])): ?><div>Remark: <?php echo safeVal(ucfirst($exam['remark'])); ?></div><?php endif; ?>
                <?php else: ?>
                    <div class="empty">No exam results found for this student.</div>
                <?php endif; ?>
            </div>
        </div>

        <?php if (!empty($marksBreakdown)): ?>
            <div class="marks">
                <?php foreach ($marksBreakdown as $subject => $score): ?>
                    <div class="card">
                        <div class="label"><?php echo safeVal(is_string($subject) ? $subject : 'Score'); ?></div>
                        <div class="value"><?php echo is_numeric($score) ? number_format((float)$score, 2) : safeVal($score); ?></div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php elseif ($exam): ?>
            <div class="card empty">No subject breakdown available.</div>
        <?php endif; ?>
    </div>
</body>
</html>
